<?php
namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use app\models\LoginForm;
use app\models\User;
use app\models\Order;
use app\models\Category;
use app\models\Product;
use yii\web\Response;
use yii\helpers\Json;

class SiteController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['logout', 'orders', 'create-order', 'view-order', 'admin'],
                'rules' => [
                    [
                        'actions' => ['logout', 'orders', 'create-order', 'view-order'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                    [
                        'actions' => ['admin'],
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return Yii::$app->user->identity->role === 'admin';
                        },
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    public function actions()
    {
        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
        ];
    }

    // Главная страница
    public function actionIndex()
    {
        return $this->render('index');
    }

    // Регистрация
    public function actionRegister()
    {
        if (!Yii::$app->user->isGuest) {
            return $this->goHome();
        }

        $model = new User();
        $model->scenario = User::SCENARIO_REGISTER;

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            return $this->redirect(['login']);  // без flash
        }

        return $this->render('register', [
            'model' => $model,
        ]);
    }
    // Авторизация
    public function actionLogin()
    {
        if (!Yii::$app->user->isGuest) {
            return $this->goHome();
        }

        $model = new LoginForm();
        if ($model->load(Yii::$app->request->post()) && $model->login()) {
            // Если админ -> редирект в админку
            if (Yii::$app->user->identity->role === 'admin') {
                return $this->redirect(['admin']);
            }
            return $this->redirect(['orders']);
        }

        return $this->render('login', [
            'model' => $model,
        ]);
    }

    // Выход
    public function actionLogout()
    {
        Yii::$app->user->logout();
        return $this->goHome();
    }

    // Просмотр своих заявок
    public function actionOrders()
    {
        $orders = Order::find()->where(['user_id' => Yii::$app->user->id])->orderBy(['created_at' => SORT_DESC])->all();
        return $this->render('orders', [
            'orders' => $orders,
        ]);
    }

    // Создание новой заявки
    public function actionCreateOrder()
    {
        $model = new Order();
        $categories = Category::find()->all();

        if ($model->load(Yii::$app->request->post())) {
            $model->status = Order::STATUS_NEW;
            if ($model->save()) {
                return $this->redirect(['orders']);  // без flash
            }
        }

        return $this->render('create-order', [
            'model' => $model,
            'categories' => $categories,
        ]);
    }

    // AJAX: получить товары по категории
    public function actionGetProducts()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $categoryId = Yii::$app->request->post('category_id');
        if ($categoryId) {
            $products = Product::find()->where(['category_id' => $categoryId])->all();
            return $products;
        }
        return [];
    }

    // Панель администратора
    public function actionAdmin()
    {
        $orders = Order::find()->orderBy(['created_at' => SORT_DESC])->all();

        if (Yii::$app->request->isPost) {
            $orderId = Yii::$app->request->post('order_id');
            $newStatus = Yii::$app->request->post('status');
            $order = Order::findOne($orderId);
            if ($order && in_array($newStatus, [Order::STATUS_NEW, Order::STATUS_ACCEPTED, Order::STATUS_COMPLETED])) {
                $order->status = $newStatus;
                $order->save(false);
                // без flash
            }
            return $this->refresh();
        }

        return $this->render('admin', [
            'orders' => $orders,
        ]);
    }
    public function actionViewOrder($id)
    {
        $order = Order::findOne($id);
        if (!$order) {
            throw new \yii\web\NotFoundHttpException('Заявка не найдена.');
        }

        // Проверка авторизации
        if (Yii::$app->user->isGuest) {
            throw new \yii\web\ForbiddenHttpException('Доступ запрещён.');
        }

        // Проверка прав: админ или владелец
        if (Yii::$app->user->identity->role !== 'admin' && $order->user_id != Yii::$app->user->id) {
            throw new \yii\web\ForbiddenHttpException('Доступ запрещён.');
        }

        return $this->render('view-order', [
            'order' => $order,
        ]);
    }
}